              
/*****************************************************************************
*                           Freescale Semiconductor                          *
*                   Brazilian Semiconductor Tecnology Center                 *
*                                                                            *
*  Project    : MC56F8006 Modular Pixel Matrix                               *
*  Version    : 0.1                                                          *
*  Date       : 27/Apr/2009                                                  *
*  Engineers  : Alexandre Dias, Bruno Bastos                                 *
*               Humberto Carvalho, Leonardo Mangiapelo                       *
*               Renato Frias                                                 *
*                                                                            *
* -------------------------------------------------------------------------- *
* THIS SOFTWARE IS PROVIDED BY FREESCALE "AS IS" AND ANY EXPRESSED OR        *
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES  *
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.    *
* IN NO EVENT SHALL FREESCALE OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,  *
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES         *
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR         *
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)         *
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,        * 
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING      *
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF             *
* THE POSSIBILITY OF SUCH DAMAGE.                                            *
* -------------------------------------------------------------------------- *
*  Description:                                                              *
* This software complements AN3815 - Implementing a Modular High Brightness  *
* RGB LED Network. The aim of the code is to proof the concept described on  *
* the application note document. It also serves as a starting point for      * 
* applications with MC56F8006 and some of its peripherals                    *
* -------------------------------------------------------------------------- *
*                                                                            *
*  File: iic.c	                                                             *
*                                                                            *
*    MC56F8006 IIC												 			 * 
*                                                                            *
******************************************************************************/

/************************* Project Include Files *****************************/

#ifndef __iic_c
#define __iic_c

#include "iic.h"
#include "rs485.h"

/********************************* Defines ***********************************/

#define EnableIntrp()	 asm(moveu.w #$0000,sr)
  
/****************************** Prototypes ***********************************/

vint8_t *txp = NULL;	/* pointer to tx buf */
vint8_t *rxp = NULL;	/* pointer to rx buf */
vint8_t *base_txBuf = NULL;
vint8_t *base_rxBuf = NULL;
vint16_t LastRdOffst; 
vint16_t LastWrOffst;

int8_t TData[I2C_TX_BUF_SIZE] = "0123456789";
int8_t RData[I2C_RX_BUF_SIZE] = "";

/*********************** Constants and Variables *****************************/

union
{
    unsigned int Word;

    struct
    {
        unsigned char tx2;
       unsigned char tx1;
    } TX_struct;
} TX_union;

#define TX1 TX_union.TX_struct.tx1
#define TX2 TX_union.TX_struct.tx2
#define TXW TX_union.Word

unsigned char RX = 0;
unsigned char *RX2;

unsigned char Data_Received = 0;
unsigned char Pending_TX1 = 0;
unsigned char Pending_TX2 = 0;

unsigned long Sum = 0;
unsigned int Max  = 0;
unsigned int Min  = 4095;

unsigned char State = 20;

unsigned int Samples = 0;
unsigned int Remaining = 0;

volatile char rxbuffer[7];
volatile char rx_buf_read_pointer=0;
volatile char tempData=0;

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

void srw (void)
{
    unsigned char RX_dummy;

    if(I2C_SR & I2C_SR_SRW) // Slave transmit?                               //D-decision
    {
        I2C_CR1 |= I2C_CR1_TX;//IICC_TX = 0b1; // I2C: TX mode
        I2C_DATA = TX1;   //IICD = TX1; // Load byte to transmit
        TX1 = 232 + State; // Load error code = "65000 + State"

        Pending_TX1 = 0;
    }
    else
    {
        I2C_CR1 &= ~I2C_CR1_TX;//IICC_TX = 0b0; // I2C: RX mode                                   //2-action
        RX_dummy = I2C_DATA; // Discard transmitted byte                     //3-action
    }
}

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

 void iic_init(void){
  
  IIC_init_drv(TData,10,RData,11);
  ENABLE_IIC();
  EnableIntrp();
  
  SET_7BIT_MODE(); /* set 7 bits addressing mode */

  SET_RECEIVE_MODE(); /* select slave mode */
  ENABLE_IIC_INTERRUPT();
      
  if(I2C_CR2 & I2C_CR2_ADEXT){ 
  	SET_SLAVE_10BIT_ADDR(SLAVE_10BIT_ADDR_1); 
  }    	
   
  else{ 
  	SET_SLAVE_7BIT_ADDR(SLAVE_7BIT_ADDR_1); /* set slave address */ 
  }      
  
  DISABLE_GENERAL_CALL(); /* not use general call */    
   
  ENABLE_IIC_INTERRUPT();
}


/*
 * IIC_init_drv: Initialize the internal driver structure and configure the GPIOB0/1
 *               pins as SCL/SDA, also enable the clock to I2C.
 *
 *
 * Parameters: 
 * txBuf  - pointer to the transmit buffer which contains data to be sent
 * txSize - the size of the data to be sent in bytes
 * rxBuf  - pointer to the receive buffer to store the received data
 * rxSize - the size of the data to receive in bytes
 *
 * Return : none
 */

void IIC_init_drv( int8_t* txBuf, int16_t txSize, int8_t* rxBuf, int16_t rxSize)
{
	base_txBuf = txp = txBuf;
	base_rxBuf = rxp = rxBuf;
	LastRdOffst = rxSize;
	LastWrOffst = txSize;
	
    SIM_GPSA    = SIM_GPSA|0x0080;
	SIM_GPSB0   = SIM_GPSB0|0x1800;
    
    GPIOA_PER |= GPIOA_PER_PER_6;
    GPIOB_PER |= GPIOB_PER_PER_4;
	
	/* Enable clock to I2C */
	SIM_PCE |= SIM_PCE_I2C;
	
}

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

#pragma interrupt alignsp saveall 
void IIC_SMBusIsr (void)//FIIC_SMBusIsr
{
	IIC_SMBusIsrCallback();
}

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/
 
#pragma interrupt alignsp called
void IIC_SMBusIsrCallback (void)
{
	if(!(I2C_SMB_CSR & I2C_SMB_CSR_SLTF))
	{
		IIC_IsrCallback();
	}
	else
	{
		CLEAR_IIC_INT();		
	}
}

/*****************************************************************************
* Funcion Name  : InitPorts
* Parameters    : none
* Returns       : none
* Notes         : 
******************************************************************************/

#pragma interrupt alignsp called
void IIC_IsrCallback (void)
{
    unsigned char RX_dummy;

    CLEAR_IIC_INT();//IICS_IICIF = 0b1;  // I2C: Clear interrupt State                    //1-action

    if (!(I2C_CR1 & I2C_CR1_MST)) // Slave mode? if (IICC_MST == 0b0)                                 //A-decision
    {
        //------ Arbitration Y
        if (I2C_SR & I2C_SR_ARBL) // Arbitration lost?                      //B-decision
        {
            I2C_SR |= I2C_SR_ARBL;//IICS_ARBL = 0b1; // Clear ARBL

            if (I2C_SR & I2C_SR_IAAS) //if (IICS_IAAS == 0b1) Addressed as slave?                //C-decision
            {
                srw ();
               
            }
        }//------ Arbitration N
        else
        {
            //------- YAAS Y (Address)
            if (I2C_SR & I2C_SR_IAAS) // if (IICS_IAAS == 0b1) Addressed as slave? 
            {
                srw ();//---- Transmit/Receiv?
                rx_buf_read_pointer=0;
            }
            else//------- YAAS N (Data)
            {
                if (I2C_CR1 & I2C_CR1_TX) // TX? if (IICC_TX == 0b1)                                //E-decision
                {
                    if (I2C_SR & I2C_SR_RXAK) // Master sent ACK? if (IICS_RXAK == 0b0) 
                    {
                        I2C_DATA = TX2; // Load byte to transmit
                        TX2 = 253; // Load error code = "65000 + State"

                        Pending_TX2 = 0;
                        
                    }
                    else
                    {
                        I2C_CR1 &= ~I2C_CR1_TX; //IICC_TX = 0b0; // I2C: RX mode
                        RX_dummy = I2C_DATA; // Discard transmitted byte
                    }
                }
                else
                {
                    //RX = IICD; // Read transmitted byte
                    tempData = I2C_DATA;
                                                               //4-action
                    rxbuffer[rx_buf_read_pointer++] = tempData;
                    Data_Received = 1;
                    //Print(tempData);
                    
                    if(tempData=='#'){
                    	
                    	rx_buf_read_pointer=0;
                    	Print(rxbuffer);
                    	
                    }
                      
                }
            }
        }
    }

}

#endif
